import { DatePicker, FormProvider, Select } from "@/src/components/RHF";
import React, { useMemo } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import Button from "@/src/components/Button";
import { useCart } from "@/src/hooks";
import { useSnackbar } from "notistack";
import { BirthDay } from "@/src/types/auth";
import { generateTimeSlots } from "@/src/utils/Helpers";
import dayjs from "dayjs";
import persianDate from "@/src/utils/persianDate";
import { FormOption } from "../../../types/general";

interface FormFields {
  date: BirthDay;
  time: FormOption;
}

interface Props {
  onAddTime: (time: string) => void;
  chefId: string;
}
const timesOptions = generateTimeSlots(24, 30);
function CartTimeForm({ onAddTime, chefId }: Props) {
  const addressSchema = useMemo(
    () =>
      Yup.object().shape({
        date: Yup.object().shape({
          day: Yup.number().moreThan(0, "فرمت تاریخ اشتباه").required(),
          month: Yup.number().moreThan(0, "فرمت تاریخ اشتباه").required(),
          year: Yup.number().moreThan(0, "فرمت تاریخ اشتباه").required(),
        }),
        time: Yup.object().shape({
          label: Yup.string().required("لطفا ساعت مدنظر خود را وارد نمایید "),
          value: Yup.string().required("لطفا ساعت مدنظر خود را وارد نمایید "),
        }),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );
  const defaultValues: FormFields = useMemo(
    () => ({
      date: {
        day: persianDate.getDate(),
        month: persianDate.getMonth(),
        year: persianDate.getYear(),
      },
      time: {
        label: "",
        value: "",
      },
    }),
    []
  );

  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(addressSchema),
    mode: "onChange",
  });

  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
    reset,
  } = methods;
  const { getOrderAvailableTime } = useCart();
  const { enqueueSnackbar } = useSnackbar();

  const onSubmit = handleSubmit(async (values) => {
    try {
      const time = values.date
        ? `${values.date.year}/${
            values.date.month >= 10
              ? values.date.month
              : `0${values.date.month}`
          }/${values.date.day} ${values.time.value}:00`
        : "";

      await getOrderAvailableTime(chefId, time);
      enqueueSnackbar("تغییرات با موفقیت انجام شد", {
        variant: "success",
      });
      reset();
      onAddTime(time);
    } catch (error) {
      console.log(error);
      enqueueSnackbar("رستوران در این ساعت سرویس نمی دهد", {
        variant: "error",
      });
    }
  });
  const start = dayjs().toDate();
  const end = dayjs().add(1, "day").toDate();
  return (
    <div className="flex  w-full p-12 ">
      <FormProvider methods={methods} onSubmit={onSubmit}>
        <div className="flex w-full flex-col items-center gap-8 lg:flex-row">
          <Select
            placeholder="ساعت ارسال"
            options={timesOptions}
            name="time"
            label=""
          />{" "}
          <DatePicker name="date" startDate={start} endDate={end} />
          <Button type="submit" isLoading={isSubmitting} disabled={!isValid}>
            بررسی
          </Button>
        </div>
      </FormProvider>
    </div>
  );
}

export default CartTimeForm;
