import {
  TextField,
  FormProvider,
  DatePicker,
  TextArea,
} from "@/src/components/RHF";
import React, { useMemo } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import { useChefs } from "@/src/hooks";
import Button from "@/src/components/Button";
import { BirthDay } from "@/src/types/auth";
import { useSnackbar } from "notistack";
import { useTranslation } from "react-i18next";
import ImageUpload from "@/src/components/ImageUploader/ImageUploader";

interface FormFields {
  name: string;
  family: string;
  nationalCode: string;
  mobile: string;
  phone?: string;
  username?: string;
  birthDay?: BirthDay;
  referralCode?: string;
  address: string;
  instagram?: string;
  telegram?: string;
  youtube?: string;
}
interface Props {
  onFormSubmit: VoidFunction;
}
function ChefRegistrationForm({ onFormSubmit }: Props) {
  const { enqueueSnackbar } = useSnackbar();
  const defaultValues: FormFields = useMemo(
    () => ({
      name: "",
      family: "",
      phone: "",
      birthDay: {
        day: 0,
        month: 0,
        year: 0,
      },
      address: "",
      mobile: "",
      nationalCode: "",
      instagram: "",
      referralCode: "",
      telegram: "",
      username: "",
      youtube: "",
    }),
    []
  );
  const registerSchema = useMemo(
    () =>
      Yup.object().shape({
        name: Yup.string().required("لطفا نام خود را وارد نمایید "),
        username: Yup.string(),
        family: Yup.string().required("لطفا نام خانوادگی خود را وارد نمایید "),
        phone: Yup.string()
          .test(
            "mobileValid",
            "لطفا شماره تلفن خود را بصورت درست وارد کنید ",
            (value) => {
              if (value) {
                return /^0[1-8]{2,}[0-9]{8,}$/.test(value);
              }
              return true;
            }
          )
          .optional(),

        birthDay: Yup.object()
          .shape({
            day: Yup.number().moreThan(0, "فرمت تاریخ اشتباه").required(),
            month: Yup.number().moreThan(0, "فرمت تاریخ اشتباه").required(),
            year: Yup.number().moreThan(0, "فرمت تاریخ اشتباه").required(),
          })
          .optional(),
        mobile: Yup.string()
          .matches(
            /^(0|0098|\+98)9(0[1-5]|[1 3]\d|2[0-2]|98)\d{7}$/,
            "لطفا شماره موبایل خود را بصورت درست وارد کنید "
          )
          .required("لطفا شماره موبایل خود را بصورت درست وارد کنید "),
        address: Yup.string().required("لطفا آدرس خود را وارد نمایید "),
        nationalCode: Yup.string()
          .matches(/^[0-9]*$/, "لطفا کد ملی خود را بصورت درست وارد کنید ")
          .length(10, "کد ملی نامعتبر")
          .required("لطفا کد ملی  خود را وارد نمایید "),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );
  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(registerSchema),
    mode: "onChange",
  });
  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
  } = methods;
  const { ChefRegistration } = useChefs();

  const onSubmit = handleSubmit(async (data) => {
    try {
      await ChefRegistration(data);
      onFormSubmit();
      enqueueSnackbar("ثبت نام با موفقیت انجام شد", {
        variant: "success",
      });
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت اطلاعات", {
        variant: "error",
      });
    }
  });
  const { t } = useTranslation();

  return (
    <div className="flex flex-col ">
      <FormProvider methods={methods} onSubmit={onSubmit}>
        <div className="grid grid-cols-1 gap-16  lg:grid-cols-3">
          <TextField fullWidth type="text" name="name" label="نام*" />
          <TextField
            fullWidth
            type="text"
            name="family"
            label="نام خانوادگی*"
          />
          <TextField
            fullWidth
            type="text"
            name="nationalCode"
            inputMode="decimal"
            label="کدملی*"
          />
          <TextField
            fullWidth
            type="text"
            name="mobile"
            label="تلفن همراه*"
            inputMode="decimal"
          />{" "}
          <TextField
            fullWidth
            type="text"
            name="phone"
            inputMode="decimal"
            label="تلفن ثابت"
          />{" "}
          <TextField fullWidth type="text" name="username" label={`${t("store_name")}*`} />{" "}
          <TextField
            fullWidth
            type="text"
            name="referralCode"
            label=" کدمعرف"
          />
          <div className="lg:col-span-2">
            <DatePicker name="birthDay" title="تاریخ تولد*" />
          </div>{" "}
        </div>
        <hr className="my-16 border border-black/20 " />
        <div className="grid grid-cols-1 gap-16 my-15 lg:grid-cols-2">
          <ImageUpload name="national_id_image" label={`${t("national_id_image")}*`} />
          <ImageUpload name="business_license_image" label={`${t("business_license_image")}*`} />

        </div>

        <div className="grid grid-cols-1 gap-8 pb-16 lg:grid-cols-2">
          <TextArea
            fullWidth
            name="address"
            label={`${t("address")}*`}
            placeholder="آدرس را بصورت کامل وارد کنید"
            rows={8}
          ></TextArea>
          <div className="flex flex-col gap-8 ">
            <p className="text-h4">شبکه های اجتماعی (اختیاری)</p>
            <TextField
              fullWidth
              type="text"
              name="instagram"
              placeholder="اینستاگرام"
            />{" "}
            <TextField
              fullWidth
              type="text"
              name="telegram"
              placeholder="کانال تلگرام"
            />{" "}
            <TextField
              fullWidth
              type="text"
              name="youtube"
              placeholder=" یوتیوب"
            />{" "}
          </div>
        </div>
        <div className="pb-16 lg:pb-0">
          <Button
            type="submit"
            fullWidth
            isLoading={isSubmitting}
            disabled={!isValid}
          >
            {t("registration")}
          </Button>
        </div>
      </FormProvider>
    </div>
  );
}

export default ChefRegistrationForm;
