"use client";

import { ChefDetail } from "@/src/types/Chef";
import React, { useEffect, useState } from "react";
import { Modal } from "@/src/components/Modal";
import SingleRestaurantBanner from "./SingleRestaurantBanner";
import SingleRestaurantTabs from "./SingleRestaurantTabs";
import DayTimePickerForm, {
  DayTimePickerDefaultValues,
} from "@/src/components/Share/DayTimePickerForm";
import { useAuthStore, useCartStore } from "@/src/store";
import { ApiSetAddressByLocation } from "@/src/services/httpServices/CartServices/types";
import { getDeliveryMethodByTimeService } from "@/src/services/httpServices/CartServices";
import DeliveryOptionsList from "../../Cart/CardFinal/components/DeliveryOptionsList";
import UseOrderAddressByParamQuery from "@/src/hooks/Queries/UseOrderAddressByParamQuery";
import RHFSelect from "@/src/components/RHF/Select";
import { FormProvider, useForm } from "react-hook-form";
import { parseAsString, useQueryStates } from "next-usequerystate";
import { useSnackbar } from "notistack";
import useDeliveryMethodByTime, {
  DeliveryMethod,
} from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { convertEnglishWeekdayToPersian } from "@/src/utils/Helpers";
import { ArrowLeft2, Edit } from "iconsax-react";
import AddressForm from "../../Profile/profileTabs/Address/AddressForm";
import AddressPrimary from "../../Profile/profileTabs/Address/AddressPrimary";
import { useCart } from "@/src/hooks";
import useAreaManagementAddressQuery from "@/src/hooks/Queries/UseAreaManagementAddressQuery";
import { useSelectionStore } from "@/src/store/useSelectionStore";
import { useRestaurantSelection } from "../../useRestaurantSelection";
import UseChefPopupAlarmsQuery from "@/src/hooks/Queries/UseChefPopupAlaramQuery";
import { getTehranTimeHHMMSS } from "@/src/utils/persianDate";

interface Props {
  chefData: ChefDetail;
  available: boolean;
}

function SingleRestaurantContents({ chefData, available }: Props) {
  const [availableModal, setAvailableModal] = useState(false);
  const [availableChef, setAvailable] = useState(true);
  const [deliveryOptions, setDeliveryOptions] = useState<DeliveryMethod[]>([]);
  const [isLoadingDeliveryMethod, setIsLoadingDeliveryMethod] = useState(false);
  const [unsavedAddress, setUnsavedAddress] = useState<string | null>();
  const [coords, setCoords] = useState<{
    lat: string | null;
    lng: string | null;
  }>({
    lat: null,
    lng: null,
  });

  const { setDeliveryTime, deliveryTime } = useRestaurantSelection(chefData.id);

  useEffect(() => {
    const unsavedAddressRaw = localStorage.getItem("unsavedAddress");
    if (unsavedAddressRaw) {
      try {
        const parsed = JSON.parse(unsavedAddressRaw);
        setCoords({
          lat: parsed.lat ?? null,
          lng: parsed.lng ?? null,
        });
      } catch (e) {
        console.error("Invalid unsavedAddress in localStorage", e);
      }
    }
  }, []);

  const { selectedDeliveryId, setSelectedDeliveryId } = useRestaurantSelection(
    chefData.id
  );

  useEffect(() => {
    if (selectedDeliveryId !== null) {
      localStorage.setItem("selectedDeliveryId", String(selectedDeliveryId));
    }
  }, [selectedDeliveryId]); // Dependency on selectedDeliveryId
  const [showAddAddress, setShowAddAddressModal] = useState(false);
  const [showAddAddressUnLogedUser, setShowAddAddressUnLogedUser] =
    useState(false);

  const { authUser } = useAuthStore();
  const [queryFilters, setQueryFilters] = useQueryStates({
    time: parseAsString.withDefault(""),
    day: parseAsString.withDefault(""),
  });
  const { enqueueSnackbar } = useSnackbar();
  const [titleUserAddressInArea, setTitleUserAddressInArea] = useState({
    title: "",
    status: false,
  });

  const [showPicker, setShowPicker] = useState(false);
  const { ensureRestaurant, updateSelection, getSelection } =
    useSelectionStore();

  const [pickerDefaults, setPickerDefaults] = useState<
    DayTimePickerDefaultValues | undefined
  >();
  const [hasDeliveryTime, setHasDeliveryTime] = useState<boolean>(false);

  useEffect(() => {
    const exists = deliveryTime !== null;
    setHasDeliveryTime(exists);
  }, [deliveryTime]);

  const [addressParams, setAddressParams] = useState<ApiSetAddressByLocation>({
    branch_id: chefData.id,
  });

  const {
    data: addressOrder,
    error: addressError,
    isLoading: isLoadingAddressOrder,
    refetchWithParamsOrderAreaAddress,
  } = UseOrderAddressByParamQuery();
  const {
    data: deliveryMethod,
    isSuccess,
    isFetching,
  } = useDeliveryMethodByTime(addressParams);

  const {
    data: popupAlarms,
    error: popupAlarmsError,
    isLoading: isLoadingPopupAlarms,
  } = UseChefPopupAlarmsQuery(chefData.id);
  const [showPopupAlarms, setShowPopupAlarms] = useState(false);
  const selectedFromStore = getSelection(chefData.id)?.selectedAddressId;
  const selectedAddressId = selectedFromStore ?? authUser.defaultAddress;

  const getTodayKey = () => {
    // produce ISO-like YYYY-MM-DD in a locale-safe way
    // using 'en-CA' returns YYYY-MM-DD format in most environments
    return new Date().toLocaleDateString("en-CA");
  };

  const makeLocalStorageKey = (id: string | number) =>
    `popup-alarm-shown:${id}`;

  const hasShownToday = (id: string | number) => {
    try {
      if (typeof window === "undefined") return false;
      const key = makeLocalStorageKey(id);
      const value = localStorage.getItem(key);
      return value === getTodayKey();
    } catch {
      return false;
    }
  };

  // mark alarm as shown today
  const markShownToday = (id: string | number) => {
    try {
      if (typeof window === "undefined") return;
      const key = makeLocalStorageKey(id);
      localStorage.setItem(key, getTodayKey());
    } catch {
      /* ignore storage failures */
    }
  };

  // effect: when popupAlarms changes, show modal only if not shown today
  useEffect(() => {
    const alarm = popupAlarms?.[0];
    if (!alarm) {
      setShowPopupAlarms(false);
      return;
    }

    const id = alarm.id ?? alarm.title ?? JSON.stringify(alarm);

    // If not shown today, open it AND mark it shown now.
    // Marking now ensures that users who don't click close won't see it again today.
    if (!hasShownToday(id)) {
      // mark first so we don't race with other logic
      markShownToday(id);
      setShowPopupAlarms(true);
    } else {
      setShowPopupAlarms(false);
    }

    // cleanup on unmount
    return () => {
      setShowPopupAlarms(false);
    };
  }, [popupAlarms]);

  const {
    data: areaAddressData,
    error,
    isSuccess: areaAddressIsSuccess,
    isLoading,
  } = useAreaManagementAddressQuery({
    branchId: chefData?.id,
    addressId: selectedAddressId,
    lat: coords.lat ?? "",
    lng: coords.lng ?? "",
  });

  useEffect(() => {
    ensureRestaurant(chefData?.id); // writes minimal record with just id
  }, [chefData?.id, ensureRestaurant]);

  useEffect(() => {
    console.log(
      "🚀 ~ SingleRestaurantContents ~ areaAddressData?.data:",
      areaAddressData?.data
    );
    if (!areaAddressData?.data) {
      console.log("❌ No delivery_type found, skipping useEffect logic");
      return;
    }
    if (areaAddressIsSuccess && areaAddressData?.data) {
      const { day, service_time_from } = areaAddressData?.data;
      const [hourStr, minuteStr] = service_time_from.split(":");
      const dayStr = day.toLowerCase() ?? "";
      const timeStr = hourStr ? `${hourStr}:${minuteStr}:00` : "";

      if (
        areaAddressData?.address_valid &&
        areaAddressData.active &&
        areaAddressData.open_branch
      ) {
        const currentTime = getTehranTimeHHMMSS(); // "20:39:00"

        setPickerDefaults({
          day: day.toLowerCase(),
          hour: parseInt(currentTime.slice(0, 2), 10),
          minute:
            parseInt(currentTime.slice(3, 5), 10) === 0
              ? 1
              : parseInt(currentTime.slice(3, 5), 10),
        });

        setQueryFilters({
          day: day.toLowerCase(),
          time: currentTime,
        });

        setAddressParams({
          branch_id: chefData?.id,
          day: day.toLowerCase(),
          time: currentTime,
        });

        setDeliveryTime({ day: day.toLowerCase(), time: currentTime ,currentTime: true});
      } else {
        setPickerDefaults({
          day: day.toLowerCase(), // e.g., "monday"
          hour: parseInt(hourStr),
          minute: parseInt(minuteStr) === 0 ? 1 : parseInt(minuteStr),
        });

        setQueryFilters({
          day: day.toLowerCase(),
          time: `${hourStr}:${minuteStr}:00`,
        });

        setAddressParams({
          branch_id: chefData?.id,
          day: day.toLowerCase(),
          time: `${hourStr}:${minuteStr}:00`,
        });

        // Save to state
        setDeliveryTime({ day: dayStr, time: timeStr });
      }
      refetchWithParamsOrderAreaAddress({
        branch_id: Number(chefData.id),
        user_id: Number(authUser.id),
        delivery_type_id: Number(areaAddressData.delivery_type.id),
      });

      setAvailable(true);
      setShowPicker(false);
    } else {
      console.log("we have no next available");
    }
    if (areaAddressIsSuccess && areaAddressData.delivery_type) {
      updateSelection(chefData.id, {
        deliveryId: areaAddressData.delivery_type.id, // Update selected delivery ID
      });
      localStorage.setItem(
        "selectedDeliveryId",
        String(areaAddressData.delivery_type.id)
      );
      const deliveryId = areaAddressData?.delivery_type?.id ?? null;
      setSelectedDeliveryId(deliveryId);
    }
    if (areaAddressIsSuccess && areaAddressData?.address_valid) {
      // setWithExpiryLocalStorage(
      //   "selectedAddressId",
      //   String(authUser?.defaultAddress),
      //   24 * 60 * 60 * 1000
      // );
      const deliveryId = areaAddressData?.delivery_type
        ? String(areaAddressData.delivery_type.id) // Convert to string
        : null; // Keep null if it's acceptable
      setSelectedDeliveryId(deliveryId as string); // Type assertion if you're sure about the type
    }
  }, [areaAddressIsSuccess, areaAddressData]);

  useEffect(() => {
    const existing = getSelection(chefData.id)?.selectedAddressId;
    if (existing) return; // already initialized, do nothing

    const fallbackAddressId = authUser?.defaultAddress ?? null;
    if (fallbackAddressId) {
      updateSelection(chefData.id, {
        selectedAddressId: String(fallbackAddressId),
      });
    } else {
      updateSelection(chefData.id, { selectedAddressId: undefined });
      localStorage.removeItem("selectedAddressId");
    }

    const deliveryId = areaAddressData?.delivery_type
      ? String(areaAddressData.delivery_type.id)
      : null;
    setSelectedDeliveryId(deliveryId as any);
  }, [
    areaAddressIsSuccess,
    areaAddressData?.address_valid,
    authUser?.defaultAddress,
  ]);

  useEffect(() => {
    if (isSuccess && deliveryMethod.delivery_methods) {
      setDeliveryOptions(deliveryMethod.delivery_methods);
      setAvailable(true);
    } else {
      setAvailable(false);
    }
  }, [isSuccess, deliveryMethod]);

  const handleTimeSubmit = async ({
    day,
    hour,
    minute,
  }: DayTimePickerDefaultValues) => {
    // Build "HH:mm:ss"
    const time = `${String(hour).padStart(2, "0")}:${String(minute ?? 0).padStart(2, "0")}:00`;

    // ✅ Persist per-restaurant via Zustand (no localStorage helper)
    setDeliveryTime({ day, time });

    try {
      setIsLoadingDeliveryMethod(true);

      // Use the same values for your API
      const response = await getDeliveryMethodByTimeService({
        branch_id: chefData.id,
        day,
        time, // already "HH:mm:ss"
      });

      if (response.delivery_methods?.length) {
        setAvailable(true);
      }

      // @ts-ignore if your type is loose
      setDeliveryOptions(response.delivery_methods);

      if (response?.error) {
        enqueueSnackbar(response.error, { variant: "error" });
      }
    } catch (error) {
      console.error("Error fetching delivery options:", error);
    } finally {
      setIsLoadingDeliveryMethod(false);
    }
  };

  const handleRadioClick = async (e: React.MouseEvent, option: any) => {
    // Temporarily clear the selected ID to force re-render
    setSelectedDeliveryId(option.id);

    localStorage.setItem("selectedDeliveryId", String(option.id));
    updateSelection(chefData.id, {
      deliveryId: option.id, // Update selected delivery ID
    });
    setAvailable(true);
    if (option.is_address) {
      const raw = localStorage.getItem("unsavedAddress");
      localStorage.setItem("has_address", "1");

      let unsavedAddress = {};
      if (raw) {
        try {
          unsavedAddress = JSON.parse(raw);
        } catch (error) {
          console.error("Invalid unsavedAddress in localStorage", error);
        }
      }

      try {
        const result = await refetchWithParamsOrderAreaAddress({
          branch_id: Number(chefData.id),
          user_id: Number(authUser.id),
          delivery_type_id: Number(option.id),
          ...unsavedAddress,
        });

        if (result.isLocationInActiveArea) {
          setTitleUserAddressInArea({
            title: "آدرس انتخاب شده در منطقه رستوران قرار دارد",
            status: true,
          });
          enqueueSnackbar("آدرس انتخاب شده در منطقه رستوران قرار دارد", {
            variant: "success",
          });

          setShowAddAddressUnLogedUser(false);
        }
        if (!result.isLocationInActiveArea && authUser.id !== 0) {
          setTitleUserAddressInArea({
            title: "آدرس انتخاب شده در منطقه رستوران قرار ندارد",
            status: false,
          });

          enqueueSnackbar("آدرس انتخاب شده در منطقه رستوران قرار ندارد ", {
            variant: "error",
          });
        }
      } catch (err: any) {
        // You can show the error message here using toast or a state
        console.error("❌ Error:", err);
      }
    } else {
      localStorage.setItem("has_address", "0");
    }
  };

  const methods = useForm({
    defaultValues: {
      address: null, // no default selected value
    },
  });

  useEffect(() => {
    if (addressOrder && authUser) {
      const selectedFromStore = getSelection(chefData.id)?.selectedAddressId;
      const selectedAddressId = selectedFromStore ?? authUser.defaultAddress;
      if (!selectedAddressId) return;

      const selectedOption = addressOrder
        .map((item) => ({
          value: item.value,
          label: `${item.label} - ${item.description.slice(0, 35)}${
            item.description.length > 35 ? "..." : ""
          } ${!item.isActive ? " - خارج از محدوده" : ""}`,
          isDisabled: !item.isActive,
        }))
        .find(
          (opt) =>
            String(opt.value) === String(selectedAddressId) && !opt.isDisabled // ✅ only allow active address
        );

      if (selectedOption) {
        methods.setValue("address", selectedOption as any, {
          shouldValidate: true,
        });
      }
    }
  }, [addressOrder, authUser, methods]);

  const selectedOption = deliveryOptions?.find(
    (item) => Number(item.id) === Number(selectedDeliveryId)
  );
  useEffect(() => {
    if (typeof window !== "undefined") {
      const item = localStorage.getItem("unsavedAddress");
      setUnsavedAddress(item);
    }
  }, []);

  useEffect(() => {
    if (typeof window !== "undefined") {
      const item = localStorage.getItem("unsavedAddress");
      setUnsavedAddress(item);
    }
  }, [deliveryOptions]);

  return (
    <div className="relative">
      <SingleRestaurantBanner chefData={chefData} available={availableChef} />
      <div className="px-4">
        <SingleRestaurantTabs
          setAvailableModal={setAvailableModal}
          deliveryMethod={deliveryMethod}
          areaAddressData={areaAddressData}
          chefData={chefData}
          available={availableChef}
          deliveryOptions={deliveryOptions}
        />
      </div>
      <Modal
        wrapperClassName="max-w-md w-full "
        open={availableModal}
        title="زمان دریافت سفارش"
        onClose={() => setAvailableModal(false)}
      >
        <div
          className={`transition-all duration-300 ${isFetching ? "pointer-events-none blur-sm" : ""}`}
        >
          {isSuccess && deliveryMethod?.next_available && (
            <div
              key={deliveryMethod.delivery_methods?.length}
              className={`text-warning-dark borde mb-4 rounded-2 bg-warning-extraLight px-8 py-6 font-iranSans text-body
                 ${deliveryOptions && deliveryOptions?.length > 0 ? "hidden" : ""}`}
            >
              <p className="text-h4 font-bold">
                زمان ارسال در حال حاضر در دسترس نیست.
              </p>
            </div>
          )}

          <div className="mb-6 flex items-center gap-4">
            <p className=" text-center text-h3 font-bold text-gray-800">
              زمان ارسال :
            </p>
            <p>
              {areaAddressData?.active && !hasDeliveryTime ? (
                "الان"
              ) : (
                <div>
                  <p className="text-h4 font-semibold text-gray-700">
                    {convertEnglishWeekdayToPersian(deliveryTime?.day || "")} -
                    <span className="mx-1"> {deliveryTime?.time}</span>
                  </p>
                </div>
              )}
            </p>

            <div className=" flex justify-center">
              <button
                onClick={() => setShowPicker((prev) => !prev)}
                className="text-primary-main rounded-half px-2 py-2 text-h4 text-primary transition-colors duration-200 "
              >
                <Edit size="16" color="#007AFF" />
              </button>
            </div>
          </div>

          {showPicker && (
            <div className="mb-6">
              <DayTimePickerForm
                loadingShow={isLoadingDeliveryMethod}
                setShowPicker={setShowPicker}
                defaultValues={pickerDefaults}
                onSubmit={handleTimeSubmit}
              />
            </div>
          )}
          <div className="mb-2 w-full border border-gray-100"></div>

          {Array.isArray(deliveryOptions) && deliveryOptions.length > 0 && (
            <>
              <p className="mb-4   text-h3 font-bold text-gray-800">
                انتخاب روش ارسال
              </p>
              <DeliveryOptionsList
                hideAddressSelectSection={true}
                deliveryData={deliveryOptions}
                selectedDeliveryId={Number(selectedDeliveryId)}
                address={undefined}
                handleRadioClick={handleRadioClick}
                setShowAddressList={function (value: boolean): void {}}
              />
            </>
          )}

          {(() => {
            if (!Array.isArray(deliveryOptions)) return null;

            if (selectedOption && deliveryOptions.length === 0)
              return <p className="text-sm text-red-500">آدرسی یافت نشد</p>;

            if (selectedOption?.is_address) {
              if (!!authUser?.id) {
                return "";
              } else {
                return (
                  <>
                    <div className=" text-sm mt-4 flex flex-col items-start gap-2 p-3 text-darkBlue">
                      {!unsavedAddress ? (
                        <p className="text-h4 text-gray-500">
                          آدرس ذخیره شده ندارید لطفا یک آدرس انتخاب کنید.
                        </p>
                      ) : (
                        ""
                      )}
                      <p
                        className={`my-4 ${titleUserAddressInArea.status ? "text-Success-main" : "text-secondary"}`}
                      >
                        {titleUserAddressInArea.title}
                      </p>
                      <button
                        onClick={() => setShowAddAddressUnLogedUser(true)}
                        className=" flex  items-center gap-4 text-h5 text-text-link"
                      >
                        انتخاب آدرس
                        <ArrowLeft2
                          size="16"
                          color="#007AFF"
                          variant="Broken"
                        />
                      </button>
                    </div>
                  </>
                );
              }
            } else {
              return <p className="text-sm text-gray-500"></p>;
            }
          })()}
          {selectedOption?.is_address &&
          addressOrder &&
          addressOrder?.length > 0 ? (
            <FormProvider {...methods}>
              <form
                onSubmit={methods.handleSubmit((formValues) => {
                  const selectedAddress: { value: string; label: string } =
                    formValues.address as any;
                  if (selectedAddress?.value) {
                    updateSelection(chefData.id, {
                      selectedAddressId: selectedAddress.value,
                    });

                    enqueueSnackbar(" آدرس با موفقیت انتخاب شد", {
                      variant: "success",
                    });
                  }

                  setAvailableModal(false);
                })}
                className="rounded-2xl shadow-md max-w-md space-y-4 bg-white"
              >
                <h2 className="text-xl mt-7 font-semibold">انتخاب آدرس</h2>

                <RHFSelect
                  name="address"
                  label="آدرس"
                  className={"text-h4"}
                  options={addressOrder
                    .slice()
                    .sort((a, b) => Number(b.isActive) - Number(a.isActive))
                    .map((item) => ({
                      value: item.value,
                      label: `${item.label} - ${item.description.slice(0, 35)}${item.description.length > 35 ? "..." : ""} ${!item.isActive ? " - خارج از محدوده" : ""}`,
                      isDisabled: !item.isActive,
                    }))}
                  fullWidth
                />
                <button
                  onClick={() => setShowAddAddressModal(true)}
                  className="flex  items-center gap-4 text-h5 text-text-link "
                >
                  افزودن آدرس جدید
                  <ArrowLeft2 size="16" color="#007AFF" variant="Broken" />
                </button>
                <div className="flex justify-end">
                  <button
                    type="submit"
                    className="rounded hover:bg-primary-dark mt-4 bg-primary px-4 py-2 text-white"
                  >
                    تایید آدرس
                  </button>
                </div>
              </form>
            </FormProvider>
          ) : (
            <div
              className={
                selectedOption?.is_address && !!authUser?.id ? "mt-4" : "hidden"
              }
            >
              <p className="text-h4 text-gray-500">آدرس یافت نشد</p>
              <button
                onClick={() => setShowAddAddressModal(true)}
                className="flex  items-center gap-4 text-h5 text-text-link "
              >
                افزودن آدرس جدید
                <ArrowLeft2 size="16" color="#007AFF" variant="Broken" />
              </button>
            </div>
          )}
          {selectedOption && !selectedOption.is_address && (
            <div className="mt-4 flex justify-end">
              <button
                onClick={() => setAvailableModal(false)}
                className="rounded hover:bg-primary-dark mt-4 bg-primary px-16 py-2 text-white"
              >
                ثبت
              </button>
            </div>
          )}
        </div>
      </Modal>

      <Modal
        title="افزودن آدرس"
        onClose={() => setShowAddAddressModal(false)}
        open={showAddAddress}
      >
        <AddressForm
          onAddAddress={async () => {
            const deliveryTypeId = selectedDeliveryId ?? 0;
            if (deliveryTypeId) {
              await refetchWithParamsOrderAreaAddress({
                branch_id: Number(chefData.id),
                user_id: Number(authUser.id),
                delivery_type_id: Number(deliveryTypeId),
              });
            } else {
              enqueueSnackbar("لطفا ابتدا روش ارسال را انتخاب کنید", {
                variant: "error",
              });
            }
            setShowAddAddressModal(false);
            setAvailableModal(true);
          }}
        />
      </Modal>

      <Modal
        title="انتخاب آدرس"
        onClose={() => setShowAddAddressUnLogedUser(false)}
        open={showAddAddressUnLogedUser}
      >
        <AddressPrimary
          shouldReloadAfterSave={false}
          onAddAddress={async ({ lat, lng }) => {
            const unsavedAddress = {
              lat: lat.toString(),
              lng: lng.toString(),
            };

            setUnsavedAddress(JSON.stringify(unsavedAddress));
            const deliveryTypeId = selectedDeliveryId ?? 0;
            if (deliveryTypeId) {
              const result = await refetchWithParamsOrderAreaAddress({
                branch_id: Number(chefData.id),
                delivery_type_id: Number(deliveryTypeId),
                lat: lat,
                lng: lng,
              });
              if (result.isLocationInActiveArea) {
                setShowAddAddressUnLogedUser(false);
                setAvailableModal(false);
                setTitleUserAddressInArea({
                  title: "آدرس انتخاب شده در منطقه رستوران قرار دارد",
                  status: true,
                });
                enqueueSnackbar("آدرس انتخاب شده در منطقه رستوران قرار دارد", {
                  variant: "success",
                });
              } else {
                setTitleUserAddressInArea({
                  title: "آدرس انتخاب شده در منطقه رستوران قرار ندارد",
                  status: false,
                });

                enqueueSnackbar(
                  "آدرس انتخاب شده در منطقه رستوران قرار ندارد ",
                  {
                    variant: "error",
                  }
                );
              }
            } else {
              enqueueSnackbar("لطفا ابتدا روش ارسال را انتخاب کنید", {
                variant: "error",
              });
            }
            setShowAddAddressUnLogedUser(false);
          }}
        />
      </Modal>

      <Modal
        wrapperClassName="min-w-[300px] min-h-[400px] lg:min-w-[400px]"
        title={popupAlarms?.[0]?.title || ""}
        onClose={() => setShowPopupAlarms(false)}
        open={showPopupAlarms}
      >
        <div className="flex flex-col gap-12">
          {popupAlarms?.[0]?.image && (
            <div className="w-full overflow-hidden rounded-md px-24">
              <img
                src={popupAlarms?.[0].image}
                alt={popupAlarms?.[0]?.title ?? "alarm image"}
                className="h-auto w-full object-cover"
              />
            </div>
          )}
          {popupAlarms?.[0]?.message && (
            <div className="px-24">
              <div 
                className="text-sm"
                dangerouslySetInnerHTML={{ __html: popupAlarms[0].message }}
              />
            </div>
          )}
        </div>
      </Modal>
    </div>
  );
}

export default SingleRestaurantContents;
