import { useEffect, useRef, useState } from "react";
import dayjs from "dayjs";

interface Props {
  EndDate: Date;
}

function useCountDown(props: Props) {
  const { EndDate } = props;

  const [countdown, setCountdown] = useState({
    days: 0,
    hours: 0,
    minutes: 0,
    seconds: 0,
  });
  const [hasCountDownFinished, setHasCountDownFinished] =
    useState<boolean>(false);
  const intervalId = useRef<NodeJS.Timeout | null>(null);
  const startCountdown = () => {
    if (intervalId.current) {
      clearInterval(intervalId.current);
    }
    intervalId.current = null;
    setHasCountDownFinished(false);
    setCountdown({
      days: 0,
      hours: 0,
      minutes: 0,
      seconds: 0,
    });
  };

  const initializeCountDown = () => {
    startCountdown();
    intervalId.current = setInterval(() => {
      const now = dayjs();
      const endDate = dayjs(EndDate);
      const diff = endDate.diff(now, "millisecond");

      if (diff <= 0 && intervalId.current) {
        clearInterval(intervalId.current);
        setHasCountDownFinished(true);
        intervalId.current = null;
        return;
      }
      const days = Math.floor(diff / (1000 * 60 * 60 * 24));
      const hours = Math.floor(
        (diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)
      );
      const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
      const seconds = Math.floor((diff % (1000 * 60)) / 1000);

      setCountdown({ days, hours, minutes, seconds });
    }, 1000);
  };

  useEffect(() => {
    initializeCountDown();
    return () => startCountdown();
  }, []);

  return {
    currentCount: countdown,
    hasCountDownFinished,
    restartCountDown: initializeCountDown,
  };
}

export default useCountDown;
