import { ApiFavoriteType } from "@/src/services/httpServices/FavoriteService/types";
import {
  ApiTicketStatusType,
  ApiTicketStatusVariant,
} from "@/src/services/httpServices/TicketServices/types";
import {
  ApiTransactionType,
  ApiTransactionVariant,
} from "@/src/services/httpServices/TransactionServices/types";
import { FavoriteType } from "@/src/types/Favorite";
import { TransactionType, TransactionVariant } from "@/src/types/Transactions";
import { FormOption, OrderDataVariant } from "@/src/types/general";
import { TicketStatusVariant, ticketStatusType } from "@/src/types/ticket";
import { ReadonlyURLSearchParams } from "next/navigation";

export const createUrl = (
  pathname: string,
  params: URLSearchParams | ReadonlyURLSearchParams
) => {
  const paramsString = params.toString();
  const queryString = `${paramsString.length ? "?" : ""}${paramsString}`;

  return `${pathname}${queryString}`;
};
export function numberWithCommas(x: number) {
  return x.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}
export function convertEnglishWeekdayToPersian(englishWeekday: string): string {
  const persianWeekdays: { [key: string]: string } = {
    saturday: "شنبه",
    sunday: "یکشنبه",
    monday: "دوشنبه",
    tuesday: "سه شنبه",
    wednesday: "چهارشنبه",
    thursday: "پنجشنبه",
    friday: "جمعه",
  };

  return persianWeekdays[englishWeekday?.toLowerCase()] ?? englishWeekday;
}
export function generateTimeSlots(
  hours: number,
  increment: number
): FormOption[] {
  const timeSlots = [];
  const timeIncrement = increment * 60; // convert minutes to seconds

  for (let hour = 0; hour < hours; hour += 1) {
    for (let minute = 0; minute < 60; minute += timeIncrement) {
      const time = new Date(1970, 0, 1, hour, minute, 0);
      const timeString = time.toTimeString().slice(0, 5); // "HH:mm" format
      timeSlots.push({ value: timeString, label: timeString });
    }
  }

  return timeSlots;
}
export function convertOrderToPersian(order: OrderDataVariant) {
  switch (order) {
    case OrderDataVariant.ASC:
      return "نزدیک ترین";
    case OrderDataVariant.DESC:
      return "جدیدترین";
    case OrderDataVariant.RATE:
      return " بالاترین امتیاز";

    default:
      return "نزدیک ترین";
  }
}
export function convertLocalFavoriteToApi(type: FavoriteType): ApiFavoriteType {
  switch (type) {
    case "chef":
      return "branch";
    case "product":
      return "product";

    default:
      return "branch";
  }
}
export function convertApiTicketStatusToLocal(
  type: ApiTicketStatusType
): ticketStatusType {
  switch (type) {
    case ApiTicketStatusVariant.OPEN:
      return TicketStatusVariant.OPEN;
    case ApiTicketStatusVariant.CLOSED:
      return TicketStatusVariant.CLOSED;
    case ApiTicketStatusVariant.WAITING:
      return TicketStatusVariant.WAITING;

    default:
      return TicketStatusVariant.WAITING;
  }
}
export const generateAuthenticatedServiceToken = (
  token?: string,
  contentType = "application/json"
) => {
  const headers = {
    headers: {
      ...(token && { Authorization: `Bearer ${token}` }),
      "Content-Type": contentType,
    },
  };

  return headers;
};
export function convertLocalTicketStatusToPersian(
  type: ticketStatusType
): string {
  switch (type) {
    case TicketStatusVariant.OPEN:
      return "باز";
    case TicketStatusVariant.CLOSED:
      return "بسته";
    case TicketStatusVariant.WAITING:
      return "در انتظار پاسخ";

    default:
      return "در انتظار پاسخ";
  }
}
export function convertApiTransactionStatusToLocal(
  type: ApiTransactionType
): TransactionType {
  switch (type) {
    case ApiTransactionVariant.CHARGE:
      return TransactionVariant.CHARGE;
    case ApiTransactionVariant.DELIVER:
      return TransactionVariant.DELIVER;
    case ApiTransactionVariant.DELIVER_PROFIT:
      return TransactionVariant.DELIVER_PROFIT;
    case ApiTransactionVariant.ORDER:
      return TransactionVariant.ORDER;
    case ApiTransactionVariant.SALES:
      return TransactionVariant.SALES;
    case ApiTransactionVariant.SALES_PROFIT:
      return TransactionVariant.SALES_PROFIT;

    default:
      return TransactionVariant.CHARGE;
  }
}
export function convertLocalOrderStatusToPersian(type: string): string {
  switch (type) {
    case "wating_payment":
      return "در انتظار پرداخت";
    case "awaiting_restaurant_approval":
      return "منتظر تائید فروشنده";
    case "preparing":
      return "درحال اماده سازی";
    case "sending":
      return "در حال ارسال";
    case "delivered":
      return "تکمیل شده";
    case "ready_to_send":
      return "آماده ارسال";
    case "returned":
      return "مرجوع شده";

    default:
      return "نامشخص";
  }
}

export function getWithExpiryLocalStorage(key: string) {
  const itemStr = localStorage.getItem(key);
  if (!itemStr) return null;

  try {
    const item = JSON.parse(itemStr);
    const now = new Date();

    if (now.getTime() > item.expiry) {
      localStorage.removeItem(key);
      return null;
    }

    return item.value;
  } catch (error) {
    localStorage.removeItem(key); // handle malformed data
    return null;
  }
}

export function setWithExpiryLocalStorage(
  key: string,
  value: any,
  ttl: number
) {
  const now = new Date();
  const item = {
    value,
    expiry: now.getTime() + ttl,
  };
  localStorage.setItem(key, JSON.stringify(item));
}

export function getWithExpiryOrPlainLocalStorage(key: string): string | null {
  const itemStr = localStorage.getItem(key);
  if (!itemStr) return null;

  try {
    const parsed = JSON.parse(itemStr);

    // ✅ Case 1: Has expiry field
    if (
      parsed &&
      typeof parsed === "object" &&
      "expiry" in parsed &&
      "value" in parsed
    ) {
      const now = Date.now();
      if (now > parsed.expiry) {
        localStorage.removeItem(key);
        return null;
      }
      return parsed.value;
    }

    // ✅ Case 2: Just a raw value (like "33")
    return parsed;
  } catch {
    // ✅ If not JSON (like a plain string "33"), return directly
    return itemStr;
  }
}
